# Python, Power BI / Fabric REST API, and PBIP

**Use Python to talk to Power BI and Microsoft Fabric via the REST API, work with PBIP (Power BI Project) folders, and run everything from Visual Studio Code.**

This guide fits the same AI-ready tenet: your semantic models and reports can be generated, deployed, or updated programmatically so they’re ready for Fabric, Copilot, and automation.

---

## What You’re Working With

| Term | What it is |
|------|------------|
| **Power BI REST API** | HTTP API for workspaces, datasets, reports, imports. Same auth and base URLs work for Fabric. |
| **Fabric REST API** | Microsoft Fabric uses the same Power BI API surface (`api.powerbi.com` or Fabric-specific endpoints). Import .pbix, manage workspaces, deploy semantic models. |
| **PBIP** | Power BI Project = a **folder** (not a single file) with JSON/text definitions (report layout, data model, metadata). Open in Power BI Desktop or use as source to build/generate .pbix for API import. |

**Typical path:** Author or generate a **PBIP folder** → open in Power BI Desktop (or build .pbix from it) → use **Python + REST API** to import that .pbix into a Fabric workspace. All of that can be run from **VS Code**.

---

## 1. Prerequisites

- **Python 3.8+** (e.g. from python.org or Windows Store).
- **Power BI Pro or Fabric capacity** and an Azure AD app with Power BI API permissions (see below).
- **Visual Studio Code** with the Python extension.

---

## 2. Azure AD App (One-Time Setup)

You need an app registration so Python can get a token for the Power BI / Fabric API.

1. Go to [Azure Portal](https://portal.azure.com) → **Azure Active Directory** → **App registrations** → **New registration**.
2. Name it (e.g. `PowerBI-Python`) → Register.
3. **API permissions** → Add permission → **Power BI Service** → **Delegated** (or Application if using client secret):
   - `Dataset.ReadWrite.All`
   - `Report.ReadWrite.All`
   - `Workspace.ReadWrite.All`
4. Grant admin consent if required.
5. Note:
   - **Application (client) ID**
   - **Directory (tenant) ID**

For **interactive (device/browser) login** you don’t need a client secret. For unattended scripts, add a **client secret** under Certificates & secrets.

---

## 3. Python Environment and Dependencies

Install the usual libraries for calling the API and (optionally) handling auth:

```bash
pip install msal requests
```

- **msal** – Microsoft Authentication Library (token for Power BI/Fabric).
- **requests** – HTTP calls to the REST API.

Use a virtual environment so the project stays self-contained (see “Running from VS Code” below).

---

## 4. Power BI / Fabric REST API Basics

- **Base URL (Power BI):** `https://api.powerbi.com/v1.0/myorg`
- **Scopes:** `https://analysis.windows.net/powerbi/api/.default` (same for Fabric when using Power BI APIs)
- **Import .pbix:** `POST https://api.powerbi.com/v1.0/myorg/groups/{workspaceId}/imports` with the .pbix file as multipart form data.

Minimal flow in Python:

1. Get a token with **MSAL** (interactive or client secret).
2. Call **GET** `/groups` to list workspaces.
3. Call **POST** `/groups/{workspaceId}/imports` with the .pbix file to deploy to Fabric/Power BI.

---

## 5. PBIP in This Flow

- **PBIP** = folder with report and model definitions (e.g. `Report/Layout`, `DataModelSchema`, `Metadata`). It’s the right format for version control and generation.
- **Open in Power BI Desktop:** File → Open → Browse → select the **PBIP folder** (not a file inside it). Enable the “Power BI Project (.pbip) save option” in Preview features if needed.
- **API path:** The REST API imports **.pbix** files, not raw PBIP. So: **PBIP → open in Desktop → Save as .pbix**, then use Python to call the Import API with that .pbix. Alternatively, use a builder script that outputs .pbix from your PBIP/definition (if you have one).

So: **PBIP = source format; .pbix = what the API accepts for import.**

---

## 6. How to Run From Visual Studio Code

### Open the project in VS Code

1. **File → Open Folder** and choose the folder that contains your Python scripts (e.g. the kit’s `examples` folder or your own repo).
2. If you have an `examples` folder with a script, open that folder so the terminal starts in the right place.

### Use a virtual environment

1. **Terminal → New Terminal** (or `` Ctrl+` ``).
2. Create a venv (one-time):
   ```bash
   python -m venv .venv
   ```
3. **Activate it:**
   - Windows (PowerShell): `.\.venv\Scripts\Activate`
   - Windows (CMD): `.venv\Scripts\activate.bat`
   - macOS/Linux: `source .venv/bin/activate`
4. Install deps in that terminal:
   ```bash
   pip install msal requests python-dotenv
   ```

### Set environment variables

1. In the project folder, create a **`.env`** file (and add `.env` to `.gitignore` so you don’t commit secrets).
2. Add at least:
   ```env
   AZURE_CLIENT_ID=your-app-client-id
   AZURE_TENANT_ID=your-tenant-id
   ```
   If using client secret:
   ```env
   AZURE_CLIENT_SECRET=your-client-secret
   ```

### Select the Python interpreter

1. **Ctrl+Shift+P** (or Cmd+Shift+P on Mac) → **Python: Select Interpreter**.
2. Pick the interpreter from your `.venv` (e.g. `.\\.venv\Scripts\python.exe` on Windows).

### Run the script

- **Run current file:** Right-click in the editor → **Run Python File in Terminal**, or use the run button (play icon) if shown.
- **Or in the Integrated Terminal** (with venv activated):
  ```bash
  python list_workspaces.py
  ```
  (Use the name of your script.)

### Optional: Run and debug

- Set a breakpoint (click in the gutter).
- **Run → Start Debugging** (F5) and choose **Python** if prompted. Execution will stop at the breakpoint.

---

## 7. Minimal Example: List Workspaces

Save this as `list_workspaces.py` in an `examples` folder (or any folder you opened in VS Code):

```python
"""List Power BI / Fabric workspaces using MSAL and REST API. Run from VS Code or: python list_workspaces.py"""
import os
from pathlib import Path

# Load .env from same folder or parent
from dotenv import load_dotenv
load_dotenv(Path(__file__).resolve().parent / ".env")

import requests

try:
    from msal import PublicClientApplication
except ImportError:
    print("Install: pip install msal requests python-dotenv")
    exit(1)

CLIENT_ID = os.getenv("AZURE_CLIENT_ID") or os.getenv("PBI_CLIENT_ID")
TENANT_ID = os.getenv("AZURE_TENANT_ID") or os.getenv("PBI_TENANT_ID") or "common"
SCOPE = ["https://analysis.windows.net/powerbi/api/.default"]
API_BASE = "https://api.powerbi.com/v1.0/myorg"

if not CLIENT_ID or not TENANT_ID:
    print("Set AZURE_CLIENT_ID and AZURE_TENANT_ID in .env")
    exit(1)

app = PublicClientApplication(client_id=CLIENT_ID, authority=f"https://login.microsoftonline.com/{TENANT_ID}")
result = app.acquire_token_interactive(scopes=SCOPE)
if "access_token" not in result:
    print("Auth failed:", result.get("error_description", result))
    exit(1)

headers = {"Authorization": f"Bearer {result['access_token']}"}
r = requests.get(f"{API_BASE}/groups", headers=headers)
if r.status_code != 200:
    print("API error", r.status_code, r.text[:300])
    exit(1)

for ws in r.json().get("value", []):
    print(ws.get("name"), "|", ws.get("id"))
```

- Put a **`.env`** in the same folder (or parent) with `AZURE_CLIENT_ID` and `AZURE_TENANT_ID`.
- From VS Code: open that folder, select the venv interpreter, run the file or run `python list_workspaces.py` in the terminal.

You should see your Power BI/Fabric workspaces listed.

---

## 8. Summary: AI Tenet and Path

- **AI tenet:** Semantic models and reports that are **ready for Q&A, Fabric, and automation**.
- **PBIP:** Author or generate reports as a project folder (version control, generation).
- **Power BI / Fabric REST API:** Deploy .pbix, manage workspaces and datasets, from scripts.
- **Python:** Glue for auth (MSAL), API calls, and any local PBIP → .pbix or generation steps.
- **VS Code:** One place to edit Python, run scripts, set .env, and use the terminal so the whole path (Python + API + PBIP) runs from one place.

For **Q&A and synonyms**, use the other guides in this kit. For **deployment and automation**, use this Python + API + PBIP path and run it from VS Code as above.
