"""
List Power BI / Fabric workspaces using MSAL and REST API.
Part of the AI-Ready Semantic Model Kit — Python + Fabric REST API + PBIP path.

Run from Visual Studio Code:
  1. Open this folder (examples) or the parent product folder in VS Code.
  2. Create/activate a venv: python -m venv .venv then .\.venv\Scripts\Activate (Windows) or source .venv/bin/activate (Mac/Linux).
  3. pip install msal requests python-dotenv
  4. Copy .env.example to .env and set AZURE_CLIENT_ID and AZURE_TENANT_ID.
  5. Python: Select Interpreter → choose .venv. Then run this file (Run Python File in Terminal) or: python list_workspaces.py
"""
import os
import sys
from pathlib import Path

# Load .env from examples folder or parent
_root = Path(__file__).resolve().parent
load_dotenv = None
try:
    from dotenv import load_dotenv
except ImportError:
    pass
if load_dotenv:
    load_dotenv(_root / ".env")
    load_dotenv(_root.parent / ".env")

import requests

try:
    from msal import PublicClientApplication
except ImportError:
    print("Install: pip install msal requests python-dotenv")
    sys.exit(1)

CLIENT_ID = os.getenv("AZURE_CLIENT_ID") or os.getenv("PBI_CLIENT_ID")
TENANT_ID = os.getenv("AZURE_TENANT_ID") or os.getenv("PBI_TENANT_ID") or "common"
SCOPE = ["https://analysis.windows.net/powerbi/api/.default"]
API_BASE = "https://api.powerbi.com/v1.0/myorg"

if not CLIENT_ID or not TENANT_ID:
    print("Set AZURE_CLIENT_ID and AZURE_TENANT_ID in .env (see .env.example)")
    sys.exit(1)

app = PublicClientApplication(
    client_id=CLIENT_ID,
    authority=f"https://login.microsoftonline.com/{TENANT_ID}",
)
result = app.acquire_token_interactive(scopes=SCOPE)
if "access_token" not in result:
    print("Auth failed:", result.get("error_description", result))
    sys.exit(1)

headers = {"Authorization": f"Bearer {result['access_token']}"}
r = requests.get(f"{API_BASE}/groups", headers=headers, timeout=30)
if r.status_code != 200:
    print("API error", r.status_code, r.text[:300])
    sys.exit(1)

data = r.json()
print("Power BI / Fabric workspaces:")
print("-" * 50)
for ws in data.get("value", []):
    print(ws.get("name"), "|", ws.get("id"))
if not data.get("value"):
    print("(none)")
print("-" * 50)
